<?php
/*
  editor.php
  Reverted Stable Version: 2025-02-12

  MAIN POINTS:
  - Right panel labeled "Documents"
  - Compare mode auto expands/collapses doc histories
  - Hide "Copy as New" icon in the versions list
  - Press Enter to rename doc
  - If doc is “unsaved,” Save button is clickable
  - On “Save,” we always create a new version in save_document.php (no data loss).
*/

require_once 'includes/functions.php';
requireLogin();
require_once 'config/db.php';

$page_title = "Web Code Editor";

$user_id = getCurrentUserId();
$stmt = $pdo->prepare("
    SELECT d.id, d.title, v.version, v.status, d.updated_at
    FROM documents d
    JOIN versions v ON d.id = v.document_id
    WHERE d.user_id = ? AND v.version = d.current_version
    ORDER BY d.updated_at DESC
");
$stmt->execute([$user_id]);
$documents = $stmt->fetchAll();

if (count($documents) > 0) {
    $mostRecent = $documents[0];
    $initialDoc = [
        'new'         => false,
        'document_id' => $mostRecent['id'],
        'version'     => $mostRecent['version'],
        'title'       => $mostRecent['title']
    ];
} else {
    $initialDoc = [
        'new'     => true,
        'title'   => 'Untitled',
        'content' => 'Hello World'
    ];
}

include 'includes/header.php';
?>

<!-- CSRF token -->
<meta name="csrf-token" content="<?php echo generateToken(); ?>">

<div class="header">
    <input type="text" class="header-title" id="pageTitle"
           value="My Code Editor (v2.0)"
           style="border:none; background:transparent; text-align:center;">
    <a href="logout.php" class="btn btn-danger btn-sm"
       style="position:absolute; right:15px; top:15px;">Logout</a>
</div>

<div class="main-container">
    <!-- Editor Panel (Left) -->
    <div class="editor-panel">
        <div class="tab-container d-flex" style="align-items:center; margin-bottom:10px;">
            <ul id="documentTabs"
                class="nav nav-tabs flex-grow-1"
                style="list-style:none; margin:0; padding:0;">
                <!-- Tabs injected by JS -->
            </ul>
            <div id="tabActions" style="margin-left:auto;">
                <button id="newDocButton" class="btn btn-link">
                    <i class="bi bi-file-earmark-plus"></i>
                </button>
                <button id="saveDocButton" class="btn btn-link" disabled>
                    <i class="bi bi-save"></i>
                </button>
            </div>
        </div>
        <div id="tabContent" class="tab-content" style="position:relative;">
            <div id="placeholderMessage" style="padding:20px; color:#888;">
                Select a document to view
            </div>
        </div>
    </div>

    <!-- Right Panel: Documents -->
    <div class="history-panel">
        <div class="d-flex justify-content-between align-items-center"
             style="margin-bottom:10px;">
            <h5 style="margin:0;">Documents</h5>
            <div>
                <button id="compareToggleBtn" class="btn btn-outline-secondary btn-sm"
                        style="margin-left:10px;">
                    <i class="bi bi-check2-square"></i> Compare Mode
                </button>
                <button id="compareSelectedBtn" class="btn btn-info btn-sm"
                        style="margin-left:5px; display:none;">
                    <i class="bi bi-arrow-left-right"></i> Compare
                </button>
            </div>
        </div>
        <?php foreach ($documents as $doc): ?>
            <div class="history-item" data-title="<?php echo htmlspecialchars($doc['title']); ?>">
                <div class="d-flex justify-content-between align-items-center">
                    <i class="bi bi-plus toggle-history"
                       data-doc-id="<?php echo $doc['id']; ?>"
                       style="cursor:pointer;"></i>
                    <span class="ms-2 open-document"
                          data-doc-id="<?php echo $doc['id']; ?>"
                          data-version="<?php echo $doc['version']; ?>"
                          style="cursor:pointer; flex-grow:1; text-align:left;">
                        <?php echo htmlspecialchars($doc['title']); ?>
                        (<?php echo ($doc['status']==='saved')
                                  ? '<span class="text-success">saved</span>'
                                  : '<span class="text-danger">unsaved</span>'; ?>)
                        (v<?php echo htmlspecialchars($doc['version']); ?>)
                    </span>
                    <i class="bi bi-x delete-document-history"
                       data-doc-id="<?php echo $doc['id']; ?>"
                       style="cursor:pointer; margin-left:5px;"></i>
                </div>
                <div class="history-versions mt-2"
                     id="history-<?php echo $doc['id']; ?>"
                     style="display:none;"
                     data-loaded="false">
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- MODALS -->
<!-- Save Success -->
<div class="modal fade" id="saveSuccessModal" tabindex="-1"
     aria-labelledby="saveSuccessModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="saveSuccessModalLabel">Save Document</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body">Document saved successfully.</div>
      <div class="modal-footer">
        <button type="button" class="btn btn-primary" id="saveSuccessOkButton">
          OK
        </button>
      </div>
    </div>
  </div>
</div>

<!-- Error Modal -->
<div class="modal fade" id="errorModal" tabindex="-1"
     aria-labelledby="errorModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="errorModalLabel">Error</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body" id="errorModalBody"></div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary"
                data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<!-- Delete Document Confirmation -->
<div class="modal fade" id="confirmDeleteDocumentModal" tabindex="-1"
     aria-labelledby="confirmDeleteDocumentModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="confirmDeleteDocumentModalLabel">
          Delete Document
        </h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body">
        Are you sure you want to permanently delete this entire document?
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary"
                data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-danger"
                id="confirmDeleteDocumentButton">Delete</button>
      </div>
    </div>
  </div>
</div>

<!-- Delete Version Confirmation -->
<div class="modal fade" id="confirmDeleteVersionModal" tabindex="-1"
     aria-labelledby="confirmDeleteVersionModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="confirmDeleteVersionModalLabel">
          Delete Version
        </h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body">
        Are you sure you want to delete this version?
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary"
                data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-danger"
                id="confirmDeleteVersionButton">Delete</button>
      </div>
    </div>
  </div>
</div>

<!-- Compare Modal -->
<div class="modal fade" id="compareModal" tabindex="-1"
     aria-labelledby="compareModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered"
       style="max-width:90%; height:90%;">
    <div class="modal-content" style="height:90%;">
      <div class="modal-header">
        <h5 class="modal-title" id="compareModalLabel">Compare Versions</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body" id="compareModalBody"
           style="height:100%; overflow:auto; line-height:1em;">
        <!-- Diff loaded from compare_versions.php -->
      </div>
    </div>
  </div>
</div>

<!-- Context Menu for Version Label -->
<div id="versionContextMenu"
     style="display:none; position:absolute; z-index:9999; background:#fff; border:1px solid #ccc; padding:5px;">
    <a href="#" id="changeVersionLabel">Change Version Label</a>
</div>

<!-- Change Version Label Modal -->
<div class="modal fade" id="versionLabelModal" tabindex="-1"
     aria-labelledby="versionLabelModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="versionLabelModalLabel">Change Version Label</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"
                aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <label for="new_version_label">New Version Label:</label>
        <input type="text" class="form-control" name="new_version_label" value="">
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary"
                data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-primary"
                id="saveVersionLabelButton">Save</button>
      </div>
    </div>
  </div>
</div>

<script>
  var initialDoc = <?php echo json_encode($initialDoc); ?>;
</script>

<?php include 'includes/footer.php'; ?>
